/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.Windows.Forms;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.Drawing; // For ToolboxBitmap
using System.Drawing.Design;
using System.Reflection;

using Borland.Eco.Subscription;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Handles;
using Borland.Eco.Services;
using Borland.Eco.UmlRt;
using Borland.Eco.Persistence;
using Borland.Eco.Windows.Forms;
using Borland.Eco.Globalization;
using Borland.Eco.Diagnostics;

namespace Borland.Eco.WinForm
{
	///<summary>
	/// Valid EcoActions for the EcoActionExtender.
	///</summary>
	public enum EcoAction
	{
		///<summary>No action is performed.</summary>
		None,
		///<summary>All persistent changes in object state are committed.</summary>
		UpdateDatabase,
		///<summary>The latest undoblock is undone.</summary>
		Undo,
		///<summary>The latest redoblock is redone.</summary>
		Redo,
		///<summary>An undo-checkpoint is set.</summary>
		SetCheckPoint,
		///<summary>The EcoSpace.Active property is toggled.</summary>
		ToggleActive,
		///<summary>A new database schema is generated.</summary>
		CreateSchema,
		///<summary>The schema is evolved to match the current model.</summary>
		EvolveSchema,
		///<summary>Shows the ecospace debugger.</summary>
		ShowDebugger
	}

//	public delegate void ListActionHandler(IObject theObject, IElementCollection elementCollection);

	///<summary>
	///<para>EcoActionExtender extends buttons with a new property EcoAction.</para>
	///<para>This allows code free execution of a number of common (and some not so common) things to do in an ecospace,
	///such as undo/redo, update database etc</para>
	///</summary>
	[ProvideProperty("EcoAction", typeof(Component))]
	[ToolboxBitmap(typeof(EcoActionExtender), "Borland.Eco.WinForm.EcoActionExtender.bmp")]
	[ToolboxItem(true)]
	[ToolboxItemFilter("System.Windows.Forms")]
	[Designer("Borland.Eco.Handles.Design.EcoActionExtenderDesigner, Borland.Eco.Handles.Design", typeof(IDesigner))]
	public class EcoActionExtender: System.ComponentModel.Component, System.ComponentModel.IExtenderProvider
	{
		private sealed class ActiveChangedAdapter: SubscriberAdapterBase
		{
			protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
			{
				if (actualSubscriber == null)
					throw new ArgumentNullException("actualSubscriber"); // do not localize
				(actualSubscriber as EcoActionExtender).ActiveChanged();
			}

			public ActiveChangedAdapter(object subscriber): base(subscriber) {}
		}

		private sealed class DirtyChangedAdapter: SubscriberAdapterBase
		{
			protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
			{
				if (actualSubscriber == null)
					 throw new ArgumentNullException("actualSubscriber"); // do not localize
				(actualSubscriber as EcoActionExtender).DirtyChanged();
			}

			public DirtyChangedAdapter(object subscriber): base(subscriber) {}
		}

		private Hashtable m_ExtendedComponents = new Hashtable();
		private ActiveChangedAdapter m_ActiveChangedAdapter;
		private DirtyChangedAdapter m_DirtyChangedAdapter;

		#region IExtenderProvider implementation
		///<summary>
		///This method determines which controls are to be considered extensible, i.e. receive new properties.
		///You may override this method in a subclass if you need additional (or other) rules to determine
		///which controls to extend.
		///</summary>		
		public virtual bool CanExtend(object extendee)
		{
			return (extendee is Button) || (extendee is MenuItem); //Currently only extend buttons
		}
		#endregion

		#region Provided properties
		#region EcoListAction
		///<summary>
		///Getter for the extended property EcoAction.
		///</summary>
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyEcoListAction")]
		[DefaultValue(EcoAction.None)]
		public EcoAction GetEcoAction(Component component)
		{
			if (m_ExtendedComponents[component] == null)
				return EcoAction.None;
			else
				return (EcoAction)m_ExtendedComponents[component];
		}

		private bool MaySetCaption(Component component)
		{
			string caption = EcoWinFormHelper.CaptionFromComponent(component);
			Control control = component as Control;
			string defaultCaption = control != null ? control.Name : string.Empty;
			return ((caption == defaultCaption) ||
				(caption == CaptionUpdateDatabase) ||
				(caption == CaptionUndo) ||
				(caption == CaptionRedo) ||
				(caption == CaptionSetCheckPoint) ||
				(caption == CaptionToggleActive) ||
				(caption == CaptionToggleDeactivate) ||
				(caption == CaptionCreateSchema) ||
				(caption == CaptionEvolveSchema) ||
				(caption == CaptionShowDebugger));
		}

		private string ActionCaption(EcoAction value)
		{
			PropertyInfo pi = this.GetType().GetProperty("Caption" + value.ToString()); // do not localize
			return (string)pi.GetValue(this, null);
		}

		///<summary>
		///This virtual method hooks up the extended control's event. The default case is to hook the protected method <see cref="OnClick"/>
		///to Click, but you may want some other event hooked, or hook to something else.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="component"/> is null.</exception>
		protected virtual void Hookup(Component component)
		{
			if (component == null) throw new ArgumentNullException("component"); // Do not localize
			Control control = component as Control;
			if (control != null)
				control.Click += new EventHandler(OnClick);
			else
			{
				MenuItem menuItem = component as MenuItem;
				if (menuItem != null)
					menuItem.Click += new EventHandler(OnClick);
			}
		}
		///<summary>
		///This virtual method unhooks up the extended controls event. The default case is to unhook the protected method <see cref="OnClick"/>
		///to Click, but if you have hooked some other event you probably want to override this too, for the sake of symmetry.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="component"/> is null.</exception>
		protected virtual void Unhook(Component component)
		{
			if (component == null) throw new ArgumentNullException("component"); // Do not localize
			Control control = component as Control;
			if (control != null) 
			{
				control.Click -= new EventHandler(OnClick);
				return;
			}
			MenuItem menuItem = component as MenuItem;
			if (menuItem != null) 
				menuItem.Click -= new EventHandler(OnClick);
		}
			
		///<summary>
		///Setter for the extended property EcoAction.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="component"/> is null.</exception>
		public void SetEcoAction(Component component, EcoAction value)
		{
			if (component == null) throw new ArgumentNullException("component"); // Do not localize
			if (value == EcoAction.None)
			{
				m_ExtendedComponents.Remove(component);
				Unhook(component);
				Control control = component as Control;
				string newCaption = control != null ? control.Name : string.Empty;
				EcoWinFormHelper.SetCaptionOnComponent(component, newCaption);
			}
			else
			{
				m_ExtendedComponents[component] = value;
				Hookup(component);
				if (MaySetCaption(component))
					EcoWinFormHelper.SetCaptionOnComponent(component, ActionCaption(value));
			}
			ActiveChanged();
		}
		#endregion

		#endregion

		#region Eventhandlers
		#region Services
		private IPersistenceService PersistenceService
		{
			get	{ return (IPersistenceService)EffectiveEcoSpace.GetEcoService(typeof(IPersistenceService)); }
		}
		private IDirtyListService DirtyListService
		{
			get
			{
				if (EffectiveEcoSpace == null)
					return null;
				return (IDirtyListService)EffectiveEcoSpace.GetEcoService(typeof(IDirtyListService));
			}
		}

		private IUndoService UndoService
		{
			get { return (IUndoService)EffectiveEcoSpace.GetEcoService(typeof(IUndoService)); }
		}
		#endregion
		private void ToggleActive(Component sender)
		{
			if (EffectiveEcoSpace != null)
				EffectiveEcoSpace.Active = !EffectiveEcoSpace.Active;
			ActiveChanged();
		}
		private void UpdateDatabase()
		{
			if ((PersistenceService != null) && (DirtyListService != null))
			{
				PersistenceService.UpdateDatabaseWithList(DirtyListService.AllDirtyObjects());
			}
		}
		private void Undo()
		{
			UndoService.UndoLatest();
		}
		private void Redo()
		{
			UndoService.RedoLatest();
		}
		private void SetCheckPoint()
		{
			UndoService.StartUndoBlock();
		}

		private IPersistenceMapperDb PersistenceMapperDb
		{
			get
			{
				if ((EffectiveEcoSpace == null) || !(EffectiveEcoSpace is DefaultEcoSpace))
					return null;
				 return ((DefaultEcoSpace)EffectiveEcoSpace).PersistenceMapper as IPersistenceMapperDb;
			}
		}

		private void CreateSchema()
		{
			try
			{
				if (PersistenceMapperDb == null)
					return;
				if (EffectiveEcoSpace.Active)
					throw new Borland.Eco.Exceptions.EcoException(FormsStringRes.sErrorSystemActive);

				PersistenceMapperDb.CreateDataBaseSchema((ITypeSystemService)EffectiveEcoSpace.GetEcoService(typeof(ITypeSystemService)), new FormBasedConfigureCleanPS());
			}
			catch (Exception e)
			{
				MessageBox.Show(e.ToString(), FormsStringRes.sCreationAborted);
			}
		}

		private void EvolveSchema()
		{
			try
			{
				if (PersistenceMapperDb == null)
					return;
				if (EffectiveEcoSpace.Active)
					throw new Borland.Eco.Exceptions.EcoException(FormsStringRes.sErrorSystemActive);
				DBEvolutionForm.EvolveDB(PersistenceMapperDb, (ITypeSystemService)EffectiveEcoSpace.GetEcoService(typeof(ITypeSystemService)), false);
			}
			catch (Exception e)
			{
				MessageBox.Show(e.ToString(), FormsStringRes.sEvolutionCaption);
			}
		}
		private void ShowEcoSpaceDebugger()
		{
			if (EffectiveEcoSpace == null)
				throw new InvalidOperationException();
			EcoSpaceDebugger debugger = new EcoSpaceDebugger(EffectiveEcoSpace);
			debugger.Show();
		}
		///<summary>
		///Performs the activity specified by the <paramref name="action"/>.
		///</summary>
		///<param name="action">The action to perform.</param>
		///<param name="component">An extended control, required to find a root handle.</param>
		public void ExecuteAction(EcoAction action, Component component)
		{
			if (!EffectiveEnabled)
				return;
			switch (action)
			{
				case EcoAction.None:
					break;
				case EcoAction.UpdateDatabase:
					UpdateDatabase();
					break;
				case EcoAction.Undo:
					Undo();
					break;
				case EcoAction.Redo:
					Redo();
					break;
				case EcoAction.SetCheckPoint:
					SetCheckPoint();
					break;
				case EcoAction.ToggleActive:
					ToggleActive(component);
					break;
				case EcoAction.CreateSchema:
					CreateSchema();
					break;
				case EcoAction.EvolveSchema:
					EvolveSchema();
					break;
				case EcoAction.ShowDebugger:
					ShowEcoSpaceDebugger();
					break;
			}
		}

		protected void OnClick(object sender, EventArgs e)
		{
			Component component = sender as Component;
			ExecuteAction(GetEcoAction(component), component);
		}
		#endregion

		private void DirtyChanged()
		{
			// Don't change during design time.
			if (Site != null)
				return;

			bool IsDirty = IsActive && (DirtyListService != null) && DirtyListService.HasDirtyObjects(); 
			foreach (Component c in m_ExtendedComponents.Keys)
			{
				if ((EcoAction)m_ExtendedComponents[c] == EcoAction.UpdateDatabase)
					EcoWinFormHelper.SetEnabledOnComponent(c, IsDirty);
			}
		}

		#region Captions

		private bool IsActive
		{
			get { return (EffectiveEcoSpace != null) && EffectiveEcoSpace.Active; }
		}

		private void ActiveChanged()
		{
			DirtyChanged();
			if (m_DirtyChangedAdapter != null)
			{
				m_DirtyChangedAdapter.Deactivate();
				m_DirtyChangedAdapter = null;
			}
			if (IsActive)
			{
				if (DirtyListService != null)
				{
					m_DirtyChangedAdapter = new DirtyChangedAdapter(this);
					DirtyListService.Subscribe(m_DirtyChangedAdapter);
				}
				UpdateConnectedComponents(EcoAction.ToggleActive, m_CaptionToggleActive, m_CaptionToggleDeactivate);
			}
			else
			{
				UpdateConnectedComponents(EcoAction.ToggleActive, m_CaptionToggleDeactivate, m_CaptionToggleActive);
			}
			DirtyChanged();
			// Don't change during design time.
			if (Site != null) return;
			bool hasRootHandle = RootHandle != null;
			foreach (Component c in m_ExtendedComponents.Keys)
			{
				EcoAction action = (EcoAction)m_ExtendedComponents[c];
				if ((action == EcoAction.CreateSchema)||
					(action == EcoAction.EvolveSchema))
					EcoWinFormHelper.SetEnabledOnComponent(c, !IsActive && hasRootHandle);
				else if ( action == EcoAction.ToggleActive)
					EcoWinFormHelper.SetEnabledOnComponent(c, hasRootHandle);
				else
					EcoWinFormHelper.SetEnabledOnComponent(c, IsActive && hasRootHandle);
			}
			DirtyChanged();
		}

		private void UpdateConnectedComponents(EcoAction action, string oldText, string newText)
		{
			foreach (Component c in m_ExtendedComponents.Keys)
			{
				if (((EcoAction)m_ExtendedComponents[c] == action) && (EcoWinFormHelper.CaptionFromComponent(c) == oldText))
					EcoWinFormHelper.SetCaptionOnComponent(c, newText);
			}
		}

		private string m_CaptionUpdateDatabase = FormsStringRes.sUpdateDatabase;
		///<summary>
		///The caption that will be set on all buttons connected to the extender, and set their EcoAction to UpdateDatabase.
		///</summary>
		[DefaultValue("Update DB")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionUpdateDatabase")]
		public string CaptionUpdateDatabase
		{
			get { return m_CaptionUpdateDatabase; }
			set { UpdateConnectedComponents(EcoAction.UpdateDatabase, m_CaptionUpdateDatabase, value); m_CaptionUpdateDatabase = value;}
		}

		private string m_CaptionUndo = FormsStringRes.sCaptionUndo;
		///<summary>
		///The caption that will be set on all buttons connected to the extender, and set their EcoAction to Undo.
		///</summary>
		[DefaultValue("Undo")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionUndo")]
		public string CaptionUndo
		{
			get { return m_CaptionUndo; }
			set { UpdateConnectedComponents(EcoAction.Undo, m_CaptionUndo, value); m_CaptionUndo = value;}
		}

		private string m_CaptionRedo = FormsStringRes.sCaptionRedo;
		///<summary>
		///The caption that will be set on all buttons connected to the extender, and set their EcoAction to Redo.
		///</summary>
		[DefaultValue("Redo")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionRedo")]
		public string CaptionRedo
		{
			get { return m_CaptionRedo; }
			set { UpdateConnectedComponents(EcoAction.Redo, m_CaptionRedo, value); m_CaptionRedo = value;}
		}

		private string m_CaptionSetCheckPoint = FormsStringRes.sCaptionSetCheckpoint;
		///<summary>
		///The caption that will be set on all buttons connected to the extender, and set their EcoAction to SetCheckPoint.
		///</summary>
		[DefaultValue("Set Checkpoint")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionSetCheckpoint")]
		public string CaptionSetCheckPoint
		{
			get { return m_CaptionSetCheckPoint; }
			set { UpdateConnectedComponents(EcoAction.SetCheckPoint, m_CaptionSetCheckPoint, value); m_CaptionSetCheckPoint = value;}
		}

		private string m_CaptionToggleActive = FormsStringRes.sCaptionToggleActive;
		///<summary>
		///The caption that will be set on all buttons connected to the extender, and set their EcoAction to ToggleActive when the system is inactive.
		///</summary>
		[DefaultValue("Activate")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionToggleActive")]
		public string CaptionToggleActive
		{
			get { return m_CaptionToggleActive; }
			set
			{
				UpdateConnectedComponents(EcoAction.ToggleActive, m_CaptionToggleActive, value);
				m_CaptionToggleActive = value;
			}
		}

		private string m_CaptionToggleDeactivate = FormsStringRes.sCaptionToggleDeactivate;
		///<summary>
		///The caption that will be set on all buttons connected to the extender, and set their EcoAction to ToggleActive when the system is active.
		///</summary>
		[DefaultValue("Deactivate")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionToggleDeactivate")]
		public string CaptionToggleDeactivate
		{
			get { return m_CaptionToggleDeactivate; }
			set
			{
				UpdateConnectedComponents(EcoAction.ToggleActive, m_CaptionToggleDeactivate, value);
				m_CaptionToggleDeactivate = value;
			}
		}

		private string m_CaptionCreateSchema = FormsStringRes.sCaptionCreateSchema;
		///<summary>
		///The caption that will be set on all buttons connected to the extender, and set their EcoAction to CreateSchema.
		///</summary>
		[DefaultValue("Create Schema")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionCreateSchema")]
		public string CaptionCreateSchema
		{
			get { return m_CaptionCreateSchema; }
			set { UpdateConnectedComponents(EcoAction.CreateSchema, m_CaptionCreateSchema, value); m_CaptionCreateSchema = value;}
		}

		private string m_CaptionEvolveSchema = FormsStringRes.sCaptionEvolveSchema;
		///<summary>
		///The caption that will be set on all buttons connected to the extender, and set their EcoAction to EvolveSchema.
		///</summary>
		[DefaultValue("Evolve Schema")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionEvolveSchema")]
		public string CaptionEvolveSchema
		{
			get { return m_CaptionEvolveSchema; }
			set { UpdateConnectedComponents(EcoAction.CreateSchema, m_CaptionEvolveSchema, value); m_CaptionEvolveSchema = value;}
		}
		private string m_CaptionShowDebugger = FormsStringRes.sCaptionShowDebugger;
		///<summary>
		///The caption that will be set on all buttons connected to the extender, and set their EcoAction to EvolveSchema.
		///</summary>
		[DefaultValue("Show ECO debugger")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionShowDebugger")]
		public string CaptionShowDebugger
		{
			get { return m_CaptionShowDebugger; }
			set { UpdateConnectedComponents(EcoAction.CreateSchema, m_CaptionShowDebugger, value); m_CaptionShowDebugger = value;}
		}
		#endregion

		#region extender's own properties
		#region Enabled property
		private bool enabled = true;
		///<summary>
		///Determins if the extender if active, i.e. if the buttons having EcoAction != null will actually perform their action.
		///</summary>
		[Browsable(true)]
		[DefaultValue(true)]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryBehaviour")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyExtenderEnabled")]
		public bool Enabled
		{
			get { return enabled; }
			set { enabled = value; }
		}
		private bool EffectiveEnabled
		{
			get { return (Enabled && (EffectiveEcoSpace != null)); }
		}
		#endregion

		#region ElementHandle
		private ElementHandle m_ElementHandle;
		///<summary>
		///The handle providing the extender with a link to the system. The system is the provider of the various EcoServices required to perform the actions.
		///The extender will follow the chain of handles until an ecospace is reached.
		///</summary>
		[Browsable(true)]
		[DefaultValue(null)]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryConnections")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyActionExtenderElementHandle")]
		[Editor("Borland.Eco.Handles.Design.RootHandleEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		public ElementHandle RootHandle
		{
			get { return m_ElementHandle; }
			set
			{
				if (m_ActiveChangedAdapter != null)
					m_ActiveChangedAdapter.Deactivate();
				m_ActiveChangedAdapter = null;
				if (m_DirtyChangedAdapter != null)
					m_DirtyChangedAdapter.Deactivate();
				m_DirtyChangedAdapter = null;
				m_ElementHandle = value;
				if (value != null)
				{
					m_ActiveChangedAdapter = new ActiveChangedAdapter(this);
					m_ElementHandle.SubscribeToActive(m_ActiveChangedAdapter);
				}
				ActiveChanged();
			}
		}
		#endregion

		#region EffectiveEcoSpace
		private EcoSpace EffectiveEcoSpace
		{
			get { return EcoWinFormHelper.ConnectedEcoSpace(RootHandle); }
		}
		#endregion
		///<summary>
		///Obsolete, for backwards compatibility. No longer does anything
		///</summary>
		[Browsable(false)]
		[Obsolete("Connect the Extender to a handle using the RootHandle property. Do not use this property.")]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public EcoSpace EcoSpace
		{
			set {}
		}
		#endregion
	}
}
